﻿using System;
using System.Diagnostics;

namespace Framework
{
    /// <summary>
    /// T型の範囲を表すクラスです。
    /// 閉区間、開区間、半開区間を表すことができます。
    /// </summary>
    public class Interval<T> where T : IComparable<T>, IComparable
    {
        internal readonly IntervalTraits traits;
        internal readonly T from;
        internal readonly T to;

        internal Interval(IntervalTraits traits, T from, T to)
        {
            this.traits = traits;
            this.from = from;
            this.to = to;
        }

        /// <summary>
        /// 引数に渡した値がこの範囲オブジェクトに含まれるかどうかを返します。
        /// 両端に制限がない場合、Tがdouble、float、HasNaNでなければ常にtrueを返します。
        /// Tがdouble、float、HasNaNの場合は、tがNaNでなければtrueを返し、NaNならfalseを返します。
        /// </summary>
        public bool Contains(T t)
        {
            switch (traits)
            {
                case IntervalTraits.OpenOpen: return from.CompareTo(t) < 0 && to.CompareTo(t) > 0;
                case IntervalTraits.OpenClose: return from.CompareTo(t) < 0 && to.CompareTo(t) >= 0;
                case IntervalTraits.CloseOpen: return from.CompareTo(t) <= 0 && to.CompareTo(t) > 0;
                case IntervalTraits.CloseClose: return from.CompareTo(t) <= 0 && to.CompareTo(t) >= 0;
                case IntervalTraits.OpenInf: return from.CompareTo(t) < 0;
                case IntervalTraits.CloseInf: return from.CompareTo(t) <= 0;
                case IntervalTraits.InfOpen: return to.CompareTo(t) > 0;
                case IntervalTraits.InfClose: return to.CompareTo(t) >= 0;
                case IntervalTraits.InfInf:
                    {
                        double? d = t as double?;
                        if (d != null)
                        {
                            return double.IsNaN(d.Value) == false;
                        }

                        float? f = t as float?;
                        if (f != null)
                        {
                            return float.IsNaN(f.Value) == false;
                        }

                        HasNaN o = t as HasNaN;
                        if (o != null)
                        {
                            return o.IsNaN == false;
                        }

                        return true;
                    }
            }

            throw new InvalidOperationException();
        }
    }

    internal enum IntervalTraits
    {
        OpenOpen,
        OpenClose,
        CloseOpen,
        CloseClose,
        OpenInf,
        CloseInf,
        InfOpen,
        InfClose,
        InfInf
    }

    /// <summary>
    /// NaN(Not a Number)を持つことを表します。
    /// </summary>
    public interface HasNaN
    {
        /// <summary>
        /// 現在のオブジェクトの状態がNaNかどうかを返します。
        /// </summary>
        bool IsNaN { get; }
    }

    public static class IntervalStringExtension
    {
        static Interval<T> Impl<T>(string fromStr, string toStr, char fromParen, char toParen, Func<string, T> converter) where T : IComparable<T>, IComparable
        {
            if (fromStr == "...")
            {
                if (toStr == "...")
                {
                    return InfInterval.Create<T>();
                }

                if (toParen == ')')
                {
                    return InfOpenInterval.To(converter(toStr));
                }

                return InfCloseInterval.To(converter(toStr));
            }
            else if (toStr == "...")
            {
                if (fromParen == '(')
                {
                    return OpenInfInterval.From(converter(fromStr));
                }

                return CloseInfInterval.From(converter(fromStr));
            }

            var from = converter(fromStr);
            var to = converter(toStr);
            if (fromParen == '(')
            {
                if (toParen == ')')
                {
                    return OpenInterval.Of(from, to);
                }

                return OpenCloseInterval.Of(from, to);
            }
            else
            {
                if (toParen == ')')
                {
                    return CloseOpenInterval.Of(from, to);
                }

                return CloseInterval.Of(from, to);
            }
        }

        static Interval<T> ToTInterval<T>(this string target, Func<string, T> converter) where T : IComparable<T>, IComparable
        {
            if (target == null)
            {
                return null;
            }

            target = target.Trim();

            var sep = target.SplitByConma();
            var fromStr = sep[0].Drop(1).Trim();
            var toStr = sep[1].DropLast(1).Trim();
            return Impl(fromStr, toStr, sep[0].First(), sep[1].Last(), converter);
        }

        /// <summary>
        /// 文字列をintの区間オブジェクトに変換します。
        /// </summary>
        public static Interval<int> ToIntInterval(this string target)
        {
            if (target == null)
            {
                throw ArgException.UnexpectedNull().butArgIsNull("target");
            }

            return target.ToTInterval(str => str.ToInt());
        }

        /// <summary>
        /// 文字列をdoubleの区間オブジェクトに変換します。
        /// </summary>
        public static Interval<double> ToDoubleInterval(this string target)
        {
            if (target == null)
            {
                throw ArgException.UnexpectedNull().butArgIsNull("target");
            }

            return target.ToTInterval(str => str.ToDouble());
        }

        /// <summary>
        /// 文字列をdecimalの区間オブジェクトに変換します。
        /// </summary>
        public static Interval<decimal> ToDecimalInterval(this string target)
        {
            if (target == null)
            {
                throw ArgException.UnexpectedNull().butArgIsNull("target");
            }

            return target.ToTInterval(str => str.ToDecimal());
        }

        /// <summary>
        /// 文字列をDateTimeの区間オブジェクトに変換します。
        /// </summary>
        public static Interval<DateTime> ToDateTimeInterval(this string target)
        {
            if (target == null)
            {
                throw ArgException.UnexpectedNull().butArgIsNull("target");
            }

            return target.ToTInterval(str => str.ToDateTime());
        }
    }

    /// <summary>
    /// 開区間を生成するためのユーティリティクラスです。
    /// </summary>
    public static class OpenInterval
    {
        /// <summary>
        /// 与えられた範囲の開区間を生成して返します。
        /// </summary>
        public static Interval<T> Of<T>(T from, T to) where T : IComparable<T>, IComparable
        {
            return new Interval<T>(IntervalTraits.OpenOpen, from, to);
        }
    }

    /// <summary>
    /// 閉区間を生成するためのユーティリティクラスです。
    /// </summary>
    public static class CloseInterval
    {
        /// <summary>
        /// 与えられた範囲の閉区間を生成して返します。
        /// </summary>
        public static Interval<T> Of<T>(T from, T to) where T : IComparable<T>, IComparable
        {
            return new Interval<T>(IntervalTraits.CloseClose, from, to);
        }
    }

    /// <summary>
    /// 半開区間を生成するためのユーティリティクラスです。
    /// </summary>
    public static class OpenCloseInterval
    {
        /// <summary>
        /// 与えられた範囲の半開区間を生成して返します。
        /// </summary>
        public static Interval<T> Of<T>(T from, T to) where T : IComparable<T>, IComparable
        {
            return new Interval<T>(IntervalTraits.OpenClose, from, to);
        }
    }

    /// <summary>
    /// 半開区間を生成するためのユーティリティクラスです。
    /// </summary>
    public static class CloseOpenInterval
    {
        /// <summary>
        /// 与えられた範囲の半開区間を生成して返します。
        /// </summary>
        public static Interval<T> Of<T>(T from, T to) where T : IComparable<T>, IComparable
        {
            return new Interval<T>(IntervalTraits.CloseOpen, from, to);
        }
    }

    /// <summary>
    /// 無限区間を生成するためのユーティリティクラスです。
    /// </summary>
    public static class OpenInfInterval
    {
        /// <summary>
        /// 与えられた点から始まる無限区間を生成して返します。
        /// </summary>
        public static Interval<T> From<T>(T from) where T : IComparable<T>, IComparable
        {
            return new Interval<T>(IntervalTraits.OpenInf, from, default(T));
        }
    }

    /// <summary>
    /// 無限区間を生成するためのユーティリティクラスです。
    /// </summary>
    public static class CloseInfInterval
    {
        /// <summary>
        /// 与えられた点から始まる無限区間を生成して返します。
        /// </summary>
        public static Interval<T> From<T>(T from) where T : IComparable<T>, IComparable
        {
            return new Interval<T>(IntervalTraits.CloseInf, from, default(T));
        }
    }

    /// <summary>
    /// 無限区間を生成するためのユーティリティクラスです。
    /// </summary>
    public static class InfOpenInterval
    {
        /// <summary>
        /// 与えられた点で終わる無限区間を生成して返します。
        /// </summary>
        public static Interval<T> To<T>(T to) where T : IComparable<T>, IComparable
        {
            return new Interval<T>(IntervalTraits.InfOpen, default(T), to);
        }
    }

    /// <summary>
    /// 無限区間を生成するためのユーティリティクラスです。
    /// </summary>
    public static class InfCloseInterval
    {
        /// <summary>
        /// 与えられた点で終わる無限区間を生成して返します。
        /// </summary>
        public static Interval<T> To<T>(T to) where T : IComparable<T>, IComparable
        {
            return new Interval<T>(IntervalTraits.InfClose, default(T), to);
        }
    }

    /// <summary>
    /// 無限区間を生成するためのユーティリティクラスです。
    /// </summary>
    public static class InfInterval
    {
        /// <summary>
        /// 無限区間を生成して返します。
        /// </summary>
        public static Interval<T> Create<T>() where T : IComparable<T>, IComparable
        {
            return new Interval<T>(IntervalTraits.InfInf, default(T), default(T));
        }
    }
}
